import Utils from "../../utils/utils";

const Autoplay = {
    run() {
        const swiper = this;
        const $activeSlideEl = swiper.slides.eq(swiper.activeIndex);
        let delay = swiper.params.autoplay.delay;
        if ($activeSlideEl.attr("data-swiper-autoplay")) {
            delay = $activeSlideEl.attr("data-swiper-autoplay") || swiper.params.autoplay.delay;
        }
        swiper.autoplay.timeout = Utils.nextTick(() => {
            if (swiper.params.autoplay.reverseDirection) {
                if (swiper.params.loop) {
                    swiper.loopFix();
                    swiper.slidePrev(swiper.params.speed, true, true);
                    swiper.emit("autoplay");
                } else if (!swiper.isBeginning) {
                    swiper.slidePrev(swiper.params.speed, true, true);
                    swiper.emit("autoplay");
                } else if (!swiper.params.autoplay.stopOnLastSlide) {
                    swiper.slideTo(swiper.slides.length - 1, swiper.params.speed, true, true);
                    swiper.emit("autoplay");
                } else {
                    swiper.autoplay.stop();
                }
            } else if (swiper.params.loop) {
                swiper.loopFix();
                swiper.slideNext(swiper.params.speed, true, true);
                swiper.emit("autoplay");
            } else if (!swiper.isEnd) {
                swiper.slideNext(swiper.params.speed, true, true);
                swiper.emit("autoplay");
            } else if (!swiper.params.autoplay.stopOnLastSlide) {
                swiper.slideTo(0, swiper.params.speed, true, true);
                swiper.emit("autoplay");
            } else {
                swiper.autoplay.stop();
            }
        }, delay);
    },
    start() {
        const swiper = this;
        if (typeof swiper.autoplay.timeout !== "undefined") return false;
        if (swiper.autoplay.running) return false;
        swiper.autoplay.running = true;
        swiper.emit("autoplayStart");
        swiper.autoplay.run();
        return true;
    },
    stop() {
        const swiper = this;
        if (!swiper.autoplay.running) return false;
        if (typeof swiper.autoplay.timeout === "undefined") return false;

        if (swiper.autoplay.timeout) {
            clearTimeout(swiper.autoplay.timeout);
            swiper.autoplay.timeout = undefined;
        }
        swiper.autoplay.running = false;
        swiper.emit("autoplayStop");
        return true;
    },
    pause(speed) {
        const swiper = this;
        if (!swiper.autoplay.running) return;
        if (swiper.autoplay.paused) return;
        if (swiper.autoplay.timeout) clearTimeout(swiper.autoplay.timeout);
        swiper.autoplay.paused = true;
        if (speed === 0 || !swiper.params.autoplay.waitForTransition) {
            swiper.autoplay.paused = false;
            swiper.autoplay.run();
        } else {
            swiper.$wrapperEl.transitionEnd(() => {
                if (!swiper || swiper.destroyed) return;
                swiper.autoplay.paused = false;
                if (!swiper.autoplay.running) {
                    swiper.autoplay.stop();
                } else {
                    swiper.autoplay.run();
                }
            });
        }
    },
};

export default {
    name: "autoplay",
    params: {
        autoplay: {
            enabled: false,
            delay: 3000,
            waitForTransition: true,
            disableOnInteraction: true,
            stopOnLastSlide: false,
            reverseDirection: false,
        },
    },
    create() {
        const swiper = this;
        Utils.extend(swiper, {
            autoplay: {
                running: false,
                paused: false,
                run: Autoplay.run.bind(swiper),
                start: Autoplay.start.bind(swiper),
                stop: Autoplay.stop.bind(swiper),
                pause: Autoplay.pause.bind(swiper),
            },
        });
    },
    on: {
        init() {
            const swiper = this;
            if (swiper.params.autoplay.enabled) {
                swiper.autoplay.start();
            }
        },
        beforeTransitionStart(speed, internal) {
            const swiper = this;
            if (swiper.autoplay.running) {
                if (internal || !swiper.params.autoplay.disableOnInteraction) {
                    swiper.autoplay.pause(speed);
                } else {
                    swiper.autoplay.stop();
                }
            }
        },
        sliderFirstMove() {
            const swiper = this;
            if (swiper.autoplay.running) {
                if (swiper.params.autoplay.disableOnInteraction) {
                    swiper.autoplay.stop();
                } else {
                    swiper.autoplay.pause();
                }
            }
        },
        destroy() {
            const swiper = this;
            if (swiper.autoplay.running) {
                swiper.autoplay.stop();
            }
        },
    },
};
